#include <stdint.h>
#include <stddef.h>

#include <system.h>
#include <usb_config.h>

#include <usb.h>
#include <usb_ch9.h>
#include <usb_device.h>
#include <usb_device_local.h>


#if !defined(USE_USB_BUS_SENSE_IO)
    #undef USB_BUS_SENSE
    #define USB_BUS_SENSE 1
#endif

#if defined(USB_DEVICE_DISABLE_DTS_CHECKING)
    #define _DTS_CHECKING_ENABLED 0
#else
    #define _DTS_CHECKING_ENABLED _DTSEN
#endif

#if !defined(self_power)
    #define self_power 0    
#endif

#if !defined(USB_MAX_NUM_CONFIG_DSC)
    #define USB_MAX_NUM_CONFIG_DSC      1
#endif

#if defined(__XC8)
    #pragma warning disable 1090
    #if __XC8_VERSION > 1300
        #pragma warning disable 1471
    #endif
#endif

typedef union
{
    uint8_t Val;
    struct __PACKED
    {
        unsigned b0:1;
        unsigned b1:1;
        unsigned b2:1;
        unsigned b3:1;
        unsigned b4:1;
        unsigned b5:1;
        unsigned b6:1;
        unsigned b7:1;
    } bits;
} uint8_t_VAL, uint8_t_BITS;


USB_VOLATILE USB_DEVICE_STATE USBDeviceState;
USB_VOLATILE uint8_t USBActiveConfiguration;
USB_VOLATILE uint8_t USBAlternateInterface[USB_MAX_NUM_INT];
volatile BDT_ENTRY *pBDTEntryEP0OutCurrent;
volatile BDT_ENTRY *pBDTEntryEP0OutNext;
volatile BDT_ENTRY *pBDTEntryOut[USB_MAX_EP_NUMBER+1];
volatile BDT_ENTRY *pBDTEntryIn[USB_MAX_EP_NUMBER+1];
USB_VOLATILE uint8_t shortPacketStatus;
USB_VOLATILE uint8_t controlTransferState;
USB_VOLATILE IN_PIPE inPipes[1];
USB_VOLATILE OUT_PIPE outPipes[1];
USB_VOLATILE uint8_t *pDst;
USB_VOLATILE bool RemoteWakeup;
USB_VOLATILE bool USBBusIsSuspended;
USB_VOLATILE USTAT_FIELDS USTATcopy;
USB_VOLATILE uint8_t endpoint_number;
USB_VOLATILE bool BothEP0OutUOWNsSet;
USB_VOLATILE EP_STATUS ep_data_in[USB_MAX_EP_NUMBER+1];
USB_VOLATILE EP_STATUS ep_data_out[USB_MAX_EP_NUMBER+1];
USB_VOLATILE uint8_t USBStatusStageTimeoutCounter;
volatile bool USBDeferStatusStagePacket;
volatile bool USBStatusStageEnabledFlag1;
volatile bool USBStatusStageEnabledFlag2;
volatile bool USBDeferINDataStagePackets;
volatile bool USBDeferOUTDataStagePackets;

volatile BDT_ENTRY BDT[BDT_NUM_ENTRIES] BDT_BASE_ADDR_TAG;

volatile CTRL_TRF_SETUP SetupPkt CTRL_TRF_SETUP_ADDR_TAG;
volatile uint8_t CtrlTrfData[USB_EP0_BUFF_SIZE] CTRL_TRF_DATA_ADDR_TAG;

extern const USB_DEVICE_DESCRIPTOR device_dsc;    
extern const uint8_t *const USB_CD_Ptr[];
extern const uint8_t *const USB_SD_Ptr[];

extern bool USER_USB_CALLBACK_EVENT_HANDLER(USB_EVENT event, void *pdata, uint16_t size);

static void USBCtrlEPService(void);
static void USBCtrlTrfSetupHandler(void);
static void USBCtrlTrfInHandler(void);
static void USBCheckStdRequest(void);
static void USBStdGetDscHandler(void);
static void USBCtrlEPServiceComplete(void);
static void USBCtrlTrfTxService(void);
static void USBCtrlTrfRxService(void);
static void USBStdSetCfgHandler(void);
static void USBStdGetStatusHandler(void);
static void USBStdFeatureReqHandler(void);
static void USBCtrlTrfOutHandler(void);
static void USBConfigureEndpoint(uint8_t EPNum, uint8_t direction);
static void USBWakeFromSuspend(void);
static void USBSuspend(void);
static void USBStallHandler(void);

#define USBAdvancePingPongBuffer(buffer) {((uint8_t_VAL*)buffer)->Val ^= USB_NEXT_PING_PONG;}
#define USBHALPingPongSetToOdd(buffer)   {((uint8_t_VAL*)buffer)->Val |= USB_NEXT_PING_PONG;}
#define USBHALPingPongSetToEven(buffer)  {((uint8_t_VAL*)buffer)->Val &= ~USB_NEXT_PING_PONG;}

void USBDeviceInit(void)
{
    uint8_t i;

    USBDisableInterrupts();

    USBClearInterruptRegister(U1EIR);  
   
    USBClearInterruptRegister(U1IR); 

    U1EP0 = 0;
    
    DisableNonZeroEndpoints(USB_MAX_EP_NUMBER);

    SetConfigurationOptions();

    for(i=0;i<(sizeof(BDT)/sizeof(BDT_ENTRY));i++)
    {
        BDT[i].Val = 0x00;
    }

    USBPingPongBufferReset = 1;                    

    U1ADDR = 0x00;                   

    USBPacketDisable = 0;           

    USBPingPongBufferReset = 0;

    while(USBTransactionCompleteIF == 1)      
    {
        USBClearInterruptFlag(USBTransactionCompleteIFReg,USBTransactionCompleteIFBitNum);
        inPipes[0].info.Val = 0;
        outPipes[0].info.Val = 0;
        outPipes[0].wCount.Val = 0;
    }
    USBStatusStageEnabledFlag1 = true;
    USBStatusStageEnabledFlag2 = true;

    USBDeferINDataStagePackets = false;
    USBDeferOUTDataStagePackets = false;
    USBBusIsSuspended = false;

    for(i = 0; i < (uint8_t)(USB_MAX_EP_NUMBER+1u); i++)
    {
        pBDTEntryIn[i] = 0u;
        pBDTEntryOut[i] = 0u;
        ep_data_in[i].Val = 0u;
        ep_data_out[i].Val = 0u;
    }
    
    pBDTEntryIn[0] = (volatile BDT_ENTRY*)&BDT[EP0_IN_EVEN];

    U1EP0 = EP_CTRL|USB_HANDSHAKE_ENABLED;        

    BDT[EP0_OUT_EVEN].ADR = ConvertToPhysicalAddress(&SetupPkt);
    BDT[EP0_OUT_EVEN].CNT = USB_EP0_BUFF_SIZE;
    BDT[EP0_OUT_EVEN].STAT.Val = _DAT0|_BSTALL;
    BDT[EP0_OUT_EVEN].STAT.Val |= _USIE;

    USBActiveConfiguration = 0;     
 
    USBDeviceState = DETACHED_STATE;
}

void USBDeviceTasks(void)
{
    uint8_t i;
    
    #if defined(USB_POLLING)
    if (USB_BUS_SENSE != 1)
    {
         U1CON = 0;             
     
         U1IE = 0;          
                
         USBDeviceState = DETACHED_STATE;

         USBClearUSBInterrupt();
         return;
    }
    if(USBDeviceState == DETACHED_STATE)
    {
        U1CON = 0;                          

        U1IE = 0;                                

        SetConfigurationOptions();

        while(!U1CONbits.USBEN){U1CONbits.USBEN = 1;}

        USBDeviceState = ATTACHED_STATE;

        #ifdef  USB_SUPPORT_OTG
            U1OTGCON |= USB_OTG_DPLUS_ENABLE | USB_OTG_ENABLE;  
        #endif
    }
	#endif  

    if(USBDeviceState == ATTACHED_STATE)
    {
        if(!USBSE0Event)
        {
            USBClearInterruptRegister(U1IR);
            #if defined(USB_POLLING)
                U1IE=0;                        
            #endif
            USBResetIE = 1;             
            USBIdleIE = 1;            
            USBDeviceState = POWERED_STATE;
        }
    }

    if(USBActivityIF && USBActivityIE)
    {
        USBClearInterruptFlag(USBActivityIFReg,USBActivityIFBitNum);

        USBWakeFromSuspend();
    }

    if(USBSuspendControl==1)
    {
        USBClearUSBInterrupt();
        return;
    }

    if(USBResetIF && USBResetIE)
    {
        USBDeviceInit();
        
        USBUnmaskInterrupts();

        USBDeviceState = DEFAULT_STATE;

        USBClearInterruptFlag(USBResetIFReg,USBResetIFBitNum);
    }

    if(USBIdleIF && USBIdleIE)
    { 
        USBSuspend();

        USBClearInterruptFlag(USBIdleIFReg,USBIdleIFBitNum);
    }

    if(USBSOFIF)
    {
        if(USBSOFIE)
        {
            USB_SOF_HANDLER(EVENT_SOF,0,1);
        }    
        USBClearInterruptFlag(USBSOFIFReg,USBSOFIFBitNum);
        
        #if defined(USB_ENABLE_STATUS_STAGE_TIMEOUTS)         
            if(USBStatusStageTimeoutCounter != 0u)
            {
                USBStatusStageTimeoutCounter--;
            }
            if(USBStatusStageTimeoutCounter == 0)
            {
                USBCtrlEPAllowStatusStage();    //Does nothing if the status stage was already armed.
            } 
        #endif
    }

    if(USBStallIF && USBStallIE)
    {
        USBStallHandler();
    }

    if(USBErrorIF && USBErrorIE)
    {
        USB_ERROR_HANDLER(EVENT_BUS_ERROR,0,1);
        USBClearInterruptRegister(U1EIR);               
    }

    if(USBDeviceState < DEFAULT_STATE)
    {
        USBClearUSBInterrupt();
        return;
    }  

    if(USBTransactionCompleteIE)
    {
        for(i = 0; i < 4u; i++)	
        {						
            if(USBTransactionCompleteIF)
            {
                USTATcopy.Val = U1STAT;
                endpoint_number = USBHALGetLastEndpoint(USTATcopy);

                USBClearInterruptFlag(USBTransactionCompleteIFReg,USBTransactionCompleteIFBitNum);

                #if (USB_PING_PONG_MODE == USB_PING_PONG__ALL_BUT_EP0) || (USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG)
                if(USBHALGetLastDirection(USTATcopy) == OUT_FROM_HOST)
                {
                    ep_data_out[endpoint_number].bits.ping_pong_state ^= 1;
                }
                else
                {
                    ep_data_in[endpoint_number].bits.ping_pong_state ^= 1;
                }
                #endif

                if(endpoint_number == 0)
                {
                    USBCtrlEPService();
                }
                else
                {
                    USB_TRANSFER_COMPLETE_HANDLER(EVENT_TRANSFER, (uint8_t*)&USTATcopy.Val, 0);
                }
            }
            else
            {
                break;	
            }
        }
    }
    USBClearUSBInterrupt();
}

void USBEnableEndpoint(uint8_t ep, uint8_t options)
{
    unsigned char* p;

    if(options & USB_OUT_ENABLED)
    {
        USBConfigureEndpoint(ep, OUT_FROM_HOST);
    }
    if(options & USB_IN_ENABLED)
    {
        USBConfigureEndpoint(ep, IN_TO_HOST);
    }
    p = (unsigned char*)(&U1EP0+ep);

    *p = options;
}

USB_HANDLE USBTransferOnePacket(uint8_t ep,uint8_t dir,uint8_t* data,uint8_t len)
{
    volatile BDT_ENTRY* handle;

    if(dir != 0)
    {
        handle = pBDTEntryIn[ep];
    }
    else
    {
        handle = pBDTEntryOut[ep];
    }
    
    if(handle == 0)
    {
        return 0;
    }

    #if (USB_PING_PONG_MODE == USB_PING_PONG__NO_PING_PONG)
        handle->STAT.Val ^= _DTSMASK;
    #elif (USB_PING_PONG_MODE == USB_PING_PONG__EP0_OUT_ONLY)
        if(ep != 0)
        {
            handle->STAT.Val ^= _DTSMASK;
        }
    #endif

    handle->ADR = ConvertToPhysicalAddress(data);
    handle->CNT = len;
    handle->STAT.Val &= _DTSMASK;
    handle->STAT.Val |= (_DTSEN & _DTS_CHECKING_ENABLED);
    handle->STAT.Val |= _USIE;

    if(dir != OUT_FROM_HOST)
    {
        USBAdvancePingPongBuffer(&pBDTEntryIn[ep]);      
    }
    else
    {
        USBAdvancePingPongBuffer(&pBDTEntryOut[ep]);     
    }
    return (USB_HANDLE)handle;
}

void USBStallEndpoint(uint8_t ep, uint8_t dir)
{
    BDT_ENTRY *p;

    if(ep == 0)
    {
        pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
        pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&SetupPkt);
        pBDTEntryEP0OutNext->STAT.Val = _DAT0|(_DTSEN & _DTS_CHECKING_ENABLED)|_BSTALL;
        pBDTEntryEP0OutNext->STAT.Val |= _USIE;
        pBDTEntryIn[0]->STAT.Val = _BSTALL; 
        pBDTEntryIn[0]->STAT.Val |= _USIE;
               
    }
    else
    {
        p = (BDT_ENTRY*)(&BDT[EP(ep,dir,0)]);
        p->STAT.Val |= _BSTALL;
        p->STAT.Val |= _USIE;
    
        #if (USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG) || (USB_PING_PONG_MODE == USB_PING_PONG__ALL_BUT_EP0)
        p = (BDT_ENTRY*)(&BDT[EP(ep,dir,1)]);
        p->STAT.Val |= _BSTALL;
        p->STAT.Val |= _USIE;
        #endif
    }
}

void USBCancelIO(uint8_t endpoint)
{
    if(USBPacketDisable == 1)
    {
    	pBDTEntryIn[endpoint]->Val &= _DTSMASK;	
    	pBDTEntryIn[endpoint]->Val ^= _DTSMASK;	
    	

        #if ((USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG) || (USB_PING_PONG_MODE == USB_PING_PONG__ALL_BUT_EP0))
        USBAdvancePingPongBuffer(&pBDTEntryIn[endpoint]);       
    
    	pBDTEntryIn[endpoint]->STAT.Val &= _DTSMASK;
    	pBDTEntryIn[endpoint]->STAT.Val ^= _DTSMASK;
        #endif
    }
}

#if defined(USB_INTERRUPT)
void USBDeviceDetach(void)
{
    {
         U1CON = 0;             
         
         U1IE = 0;          
        
         USBDeviceState = DETACHED_STATE;

          return;
    }
}
#endif  

#if defined(USB_INTERRUPT)
void USBDeviceAttach(void)
{
    if(USBDeviceState == DETACHED_STATE)
    {
        if(USB_BUS_SENSE == 1)
        {
            U1CON = 0;          
    
            U1IE = 0;                                

            SetConfigurationOptions();
    
            USBEnableInterrupts();  
   
            while(!U1CONbits.USBEN){U1CONbits.USBEN = 1;}
    

            USBDeviceState = ATTACHED_STATE;
        }
    }
}
#endif  


void USBCtrlEPAllowStatusStage(void)
{
    if(USBStatusStageEnabledFlag1 == false)
    {
        USBStatusStageEnabledFlag1 = true;
        if(USBStatusStageEnabledFlag2 == false)
        {
            USBStatusStageEnabledFlag2 = true;

            if(controlTransferState == CTRL_TRF_RX)
            {
                pBDTEntryIn[0]->CNT = 0;
                pBDTEntryIn[0]->STAT.Val = _DAT1|(_DTSEN & _DTS_CHECKING_ENABLED);        
                pBDTEntryIn[0]->STAT.Val |= _USIE;
            }
            else if(controlTransferState == CTRL_TRF_TX)
            {
                BothEP0OutUOWNsSet = false;

                #if((USB_PING_PONG_MODE == USB_PING_PONG__EP0_OUT_ONLY) || (USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG))
                    pBDTEntryEP0OutCurrent->CNT = USB_EP0_BUFF_SIZE;
                    pBDTEntryEP0OutCurrent->ADR = ConvertToPhysicalAddress(&SetupPkt);
                    pBDTEntryEP0OutCurrent->STAT.Val = _BSTALL; 
                    pBDTEntryEP0OutCurrent->STAT.Val |= _USIE;
                    BothEP0OutUOWNsSet = true;	
                #endif

                pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
                pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&SetupPkt);
                pBDTEntryEP0OutNext->STAT.Val = _USIE;           
            }
        }    
    }
}   

void USBCtrlEPAllowDataStage(void)
{
    USBDeferINDataStagePackets = false;
    USBDeferOUTDataStagePackets = false;

    if(controlTransferState == CTRL_TRF_RX) 
    {
        pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
        pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&CtrlTrfData);
        pBDTEntryEP0OutNext->STAT.Val = _DAT1|(_DTSEN & _DTS_CHECKING_ENABLED);
        pBDTEntryEP0OutNext->STAT.Val |= _USIE;
    }   
    else    
    {
		if(SetupPkt.wLength < inPipes[0].wCount.Val)
		{
			inPipes[0].wCount.Val = SetupPkt.wLength;
		}
		USBCtrlTrfTxService();  

		pBDTEntryIn[0]->ADR = ConvertToPhysicalAddress(&CtrlTrfData);
		pBDTEntryIn[0]->STAT.Val = _DAT1|(_DTSEN & _DTS_CHECKING_ENABLED);
        pBDTEntryIn[0]->STAT.Val |= _USIE;
    }     
}    

static void USBConfigureEndpoint(uint8_t EPNum, uint8_t direction)
{
    volatile BDT_ENTRY* handle;

    handle = (volatile BDT_ENTRY*)&BDT[EP0_OUT_EVEN]; 
    handle += EP(EPNum,direction,0u);    
    
    handle->STAT.UOWN = 0;  

    if(direction == OUT_FROM_HOST)
    {
        pBDTEntryOut[EPNum] = handle;
    }
    else
    {
        pBDTEntryIn[EPNum] = handle;
    }

    #if (USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG)
        handle->STAT.DTS = 0;
        (handle+1)->STAT.DTS = 1;
    #elif (USB_PING_PONG_MODE == USB_PING_PONG__NO_PING_PONG)
        //Set DTS to one because the first thing we will do
        //when transmitting is toggle the bit
        handle->STAT.DTS = 1;
    #elif (USB_PING_PONG_MODE == USB_PING_PONG__EP0_OUT_ONLY)
        if(EPNum != 0)
        {
            handle->STAT.DTS = 1;
        }
    #elif (USB_PING_PONG_MODE == USB_PING_PONG__ALL_BUT_EP0)    
        if(EPNum != 0)
        {
            handle->STAT.DTS = 0;
            (handle+1)->STAT.DTS = 1;
        }
    #endif
}

static void USBCtrlEPServiceComplete(void)
{

    USBPacketDisable = 0;

    if(inPipes[0].info.bits.busy == 0)
    {
        if(outPipes[0].info.bits.busy == 1)
        {
            controlTransferState = CTRL_TRF_RX;

            if(USBDeferOUTDataStagePackets == false)
            {
                USBCtrlEPAllowDataStage();
            }

            USBStatusStageEnabledFlag2 = false;
            USBStatusStageEnabledFlag1 = false;
        }
        else
        {
            pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
            pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&SetupPkt);
            pBDTEntryEP0OutNext->STAT.Val = _DAT0|(_DTSEN & _DTS_CHECKING_ENABLED)|_BSTALL;
            pBDTEntryEP0OutNext->STAT.Val |= _USIE;
            pBDTEntryIn[0]->STAT.Val = _BSTALL;
            pBDTEntryIn[0]->STAT.Val |= _USIE;
        }
    }
    else    
    {
		if(SetupPkt.DataDir == USB_SETUP_DEVICE_TO_HOST_BITFIELD)
		{
			controlTransferState = CTRL_TRF_TX;
			if(USBDeferINDataStagePackets == false)
            {
                USBCtrlEPAllowDataStage();
			}

            USBStatusStageEnabledFlag2 = false;
            USBStatusStageEnabledFlag1 = false;
            if(USBDeferStatusStagePacket == false)
            {
                USBCtrlEPAllowStatusStage();
            } 
		}
		else  
		{


			controlTransferState = CTRL_TRF_RX;     
			
			pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
			pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&SetupPkt);
			pBDTEntryEP0OutNext->STAT.Val = _BSTALL;
            pBDTEntryEP0OutNext->STAT.Val |= _USIE;
				
            USBStatusStageEnabledFlag2 = false;
            USBStatusStageEnabledFlag1 = false;
			if(USBDeferStatusStagePacket == false)
            {
                USBCtrlEPAllowStatusStage();
            } 
		}

    }

}

static void USBCtrlTrfTxService(void)
{
    uint8_t byteToSend;

    byteToSend = USB_EP0_BUFF_SIZE;         
    if(inPipes[0].wCount.Val < (uint8_t)USB_EP0_BUFF_SIZE)
    {
        byteToSend = inPipes[0].wCount.Val;
        
        if(shortPacketStatus == SHORT_PKT_NOT_USED)
        {
            shortPacketStatus = SHORT_PKT_PENDING;
        }
        else if(shortPacketStatus == SHORT_PKT_PENDING)
        {
            shortPacketStatus = SHORT_PKT_SENT;
        }
    }
    inPipes[0].wCount.Val -= byteToSend;

    pBDTEntryIn[0]->CNT = byteToSend;

    pDst = (USB_VOLATILE uint8_t*)CtrlTrfData;               
    if(inPipes[0].info.bits.ctrl_trf_mem == USB_EP0_ROM)   
    {
        while(byteToSend)
        {
            *pDst++ = *inPipes[0].pSrc.bRom++;
            byteToSend--;
        }
    }
    else  
    {
        while(byteToSend)
        {
            *pDst++ = *inPipes[0].pSrc.bRam++;
            byteToSend--;
        }
    }
}
static void USBCtrlTrfRxService(void)
{
    uint8_t byteToRead;
    uint8_t i;

    byteToRead = pBDTEntryEP0OutCurrent->CNT;   
    
    if(byteToRead > outPipes[0].wCount.Val)
    {
        byteToRead = outPipes[0].wCount.Val;
    }	

    outPipes[0].wCount.Val -= byteToRead;

    for(i=0;i<byteToRead;i++)
    {
        *outPipes[0].pDst.bRam++ = CtrlTrfData[i];
    }

    if(outPipes[0].wCount.Val > 0)
    {
        pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
        pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&CtrlTrfData);
        if(pBDTEntryEP0OutCurrent->STAT.DTS == 0)
        {
            pBDTEntryEP0OutNext->STAT.Val = _DAT1|(_DTSEN & _DTS_CHECKING_ENABLED);
            pBDTEntryEP0OutNext->STAT.Val |= _USIE;
        }
        else
        {
            pBDTEntryEP0OutNext->STAT.Val = _DAT0|(_DTSEN & _DTS_CHECKING_ENABLED);
            pBDTEntryEP0OutNext->STAT.Val |= _USIE;
        }
    }
    else
    {
        pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
        pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&SetupPkt);

        pBDTEntryEP0OutNext->STAT.Val = _BSTALL;
        pBDTEntryEP0OutNext->STAT.Val |= _USIE;

        if(outPipes[0].pFunc != NULL)
        {
            #if defined(__XC8)

                #pragma warning push
                #pragma warning disable 1088
                outPipes[0].pFunc();    
                #pragma warning pop
            #else
                outPipes[0].pFunc();    
            #endif
        }
        outPipes[0].info.bits.busy = 0;    

        if(USBDeferStatusStagePacket == false)
        {
            USBCtrlEPAllowStatusStage();
        }            
    }    
}

static void USBStdSetCfgHandler(void)
{
    uint8_t i;

    inPipes[0].info.bits.busy = 1;            

    DisableNonZeroEndpoints(USB_MAX_EP_NUMBER);

    memset((void*)&BDT[0], 0x00, sizeof(BDT));

    USBPingPongBufferReset = 1;                                   

	for(i = 0; i < (uint8_t)(USB_MAX_EP_NUMBER+1u); i++)
	{
		ep_data_in[i].Val = 0u;
        ep_data_out[i].Val = 0u;
	}

    memset((void*)&USBAlternateInterface,0x00,USB_MAX_NUM_INT);

    USBPingPongBufferReset = 0;

    pBDTEntryIn[0] = (volatile BDT_ENTRY*)&BDT[EP0_IN_EVEN];

    pBDTEntryEP0OutCurrent = (volatile BDT_ENTRY*)&BDT[EP0_OUT_EVEN];
    pBDTEntryEP0OutNext = pBDTEntryEP0OutCurrent;

    USBActiveConfiguration = SetupPkt.bConfigurationValue;

    if(USBActiveConfiguration == 0)
    {
        USBDeviceState = ADDRESS_STATE;
    }
    else
    {
        USB_SET_CONFIGURATION_HANDLER(EVENT_CONFIGURED,(void*)&USBActiveConfiguration,1);

        USBDeviceState = CONFIGURED_STATE;		
    }
}

static void USBStdGetDscHandler(void)
{
    if(SetupPkt.bmRequestType == 0x80)
    {
        inPipes[0].info.Val = USB_EP0_ROM | USB_EP0_BUSY | USB_EP0_INCLUDE_ZERO;

        switch(SetupPkt.bDescriptorType)
        {
            case USB_DESCRIPTOR_DEVICE:
                
                inPipes[0].pSrc.bRom = (const uint8_t*)&device_dsc;
                inPipes[0].wCount.Val = sizeof(device_dsc);
                break;
            case USB_DESCRIPTOR_CONFIGURATION:
                if(SetupPkt.bDscIndex < USB_MAX_NUM_CONFIG_DSC)
                {
                   
                    inPipes[0].pSrc.bRom = *(USB_CD_Ptr+SetupPkt.bDscIndex);
                    inPipes[0].wCount.byte.LB = *(inPipes[0].pSrc.bRom+2);
                    inPipes[0].wCount.byte.HB = *(inPipes[0].pSrc.bRom+3);
                }
				else
				{
					inPipes[0].info.Val = 0;
				}
                break;
            case USB_DESCRIPTOR_STRING:

                if(SetupPkt.bDscIndex<USB_NUM_STRING_DESCRIPTORS)
                {
                    inPipes[0].pSrc.bRom = *(USB_SD_Ptr+SetupPkt.bDscIndex);
                    inPipes[0].wCount.Val = *inPipes[0].pSrc.bRom;                    
                }
                else
                {
                    inPipes[0].info.Val = 0;
                }
                break;
            default:
                inPipes[0].info.Val = 0;
                break;
        }
    }
}

static void USBStdGetStatusHandler(void)
{
    CtrlTrfData[0] = 0;                 
    CtrlTrfData[1] = 0;

    switch(SetupPkt.Recipient)
    {
        case USB_SETUP_RECIPIENT_DEVICE_BITFIELD:
            inPipes[0].info.bits.busy = 1;

            if(self_power == 1) 
            {
                CtrlTrfData[0]|=0x01;
            }

            if(RemoteWakeup == true)
            {
                CtrlTrfData[0]|=0x02;
            }
            break;
        case USB_SETUP_RECIPIENT_INTERFACE_BITFIELD:
            inPipes[0].info.bits.busy = 1;     
            break;
        case USB_SETUP_RECIPIENT_ENDPOINT_BITFIELD:
            inPipes[0].info.bits.busy = 1;
            {
                BDT_ENTRY *p;

                if(SetupPkt.EPDir == 0)
                {
                    p = (BDT_ENTRY*)pBDTEntryOut[SetupPkt.EPNum];
                }
                else
                {
                    p = (BDT_ENTRY*)pBDTEntryIn[SetupPkt.EPNum];
                }

                if((p->STAT.UOWN == 1) && (p->STAT.BSTALL == 1))
                    CtrlTrfData[0]=0x01;    
                break;
            }
    }

    if(inPipes[0].info.bits.busy == 1)
    {
        inPipes[0].pSrc.bRam = (uint8_t*)&CtrlTrfData;       
        inPipes[0].info.bits.ctrl_trf_mem = USB_EP0_RAM;    
        inPipes[0].wCount.v[0] = 2;                           
    }
}

static void USBStallHandler(void)
{


    if(U1EP0bits.EPSTALL == 1)
    {
        if((pBDTEntryEP0OutCurrent->STAT.Val == _USIE) && (pBDTEntryIn[0]->STAT.Val == (_USIE|_BSTALL)))
        {

            pBDTEntryEP0OutCurrent->STAT.Val = _DAT0|(_DTSEN & _DTS_CHECKING_ENABLED)|_BSTALL;
            pBDTEntryEP0OutCurrent->STAT.Val |= _USIE;
        }
        U1EP0bits.EPSTALL = 0;               
    }

    USBClearInterruptFlag(USBStallIFReg,USBStallIFBitNum);
}

static void USBSuspend(void)
{
    USBActivityIE = 1;                     
    USBClearInterruptFlag(USBIdleIFReg,USBIdleIFBitNum);
    U1CONbits.SUSPND = 1;                  

    USBBusIsSuspended = true;
 
    USB_SUSPEND_HANDLER(EVENT_SUSPEND,0,0);
}

static void USBWakeFromSuspend(void)
{
    USBBusIsSuspended = false;

    USB_WAKEUP_FROM_SUSPEND_HANDLER(EVENT_RESUME,0,0);
    U1CONbits.SUSPND = 0;
    
    USBActivityIE = 0;




    while(USBActivityIF)
    {
        USBClearInterruptFlag(USBActivityIFReg,USBActivityIFBitNum);
    }  
}

static void USBCtrlEPService(void)
{
    #if defined(USB_ENABLE_STATUS_STAGE_TIMEOUTS)
        USBStatusStageTimeoutCounter = USB_STATUS_STAGE_TIMEOUT;
    #endif
	
    if((USTATcopy.Val & USTAT_EP0_PP_MASK) == USTAT_EP0_OUT_EVEN)
    {

        pBDTEntryEP0OutCurrent = (volatile BDT_ENTRY*)&BDT[(USTATcopy.Val & USTAT_EP_MASK)>>1];

        pBDTEntryEP0OutNext = pBDTEntryEP0OutCurrent;

        ((uint8_t_VAL*)&pBDTEntryEP0OutNext)->Val ^= USB_NEXT_EP0_OUT_PING_PONG;

        if(pBDTEntryEP0OutCurrent->STAT.PID == PID_SETUP)
        {
            unsigned char setup_cnt;

            for(setup_cnt = 0; setup_cnt < 8u; setup_cnt++) 
            {
                *(uint8_t*)((uint8_t*)&SetupPkt + setup_cnt) = *(uint8_t*)ConvertToVirtualAddress(pBDTEntryEP0OutCurrent->ADR);
                pBDTEntryEP0OutCurrent->ADR++;
            }    
            pBDTEntryEP0OutCurrent->ADR = ConvertToPhysicalAddress(&SetupPkt);

            USBCtrlTrfSetupHandler();
        }
        else
        {
            USBCtrlTrfOutHandler();
        }
    }
    else if((USTATcopy.Val & USTAT_EP0_PP_MASK) == USTAT_EP0_IN)
    {
        USBCtrlTrfInHandler();
    }

}

static void USBCtrlTrfSetupHandler(void)
{

    shortPacketStatus = SHORT_PKT_NOT_USED;  
    USBDeferStatusStagePacket = false;
    USBDeferINDataStagePackets = false;
    USBDeferOUTDataStagePackets = false;
    BothEP0OutUOWNsSet = false;
    controlTransferState = WAIT_SETUP;

    pBDTEntryIn[0]->STAT.Val &= ~(_USIE);     
    ((uint8_t_VAL*)&pBDTEntryIn[0])->Val ^= USB_NEXT_EP0_IN_PING_PONG;
    pBDTEntryIn[0]->STAT.Val &= ~(_USIE);      
    ((uint8_t_VAL*)&pBDTEntryIn[0])->Val ^= USB_NEXT_EP0_IN_PING_PONG;
    pBDTEntryEP0OutNext->STAT.Val &= ~(_USIE);         

    inPipes[0].info.Val = 0;
    inPipes[0].wCount.Val = 0;
    outPipes[0].info.Val = 0;
    outPipes[0].wCount.Val = 0;
    
    USBCheckStdRequest();                                               
    USB_NONSTANDARD_EP0_REQUEST_HANDLER(EVENT_EP0_REQUEST,0,0);

    USBCtrlEPServiceComplete();
}

static void USBCtrlTrfOutHandler(void)
{
    if(controlTransferState == CTRL_TRF_RX)
    {
        USBCtrlTrfRxService();	
    }
    else 
    {
        controlTransferState = WAIT_SETUP;

        if(BothEP0OutUOWNsSet == false)
        {
            pBDTEntryEP0OutNext->CNT = USB_EP0_BUFF_SIZE;
            pBDTEntryEP0OutNext->ADR = ConvertToPhysicalAddress(&SetupPkt);
            pBDTEntryEP0OutNext->STAT.Val = _DAT0|(_DTSEN & _DTS_CHECKING_ENABLED)|_BSTALL;
            pBDTEntryEP0OutNext->STAT.Val |= _USIE;
        }
        else
        {
                BothEP0OutUOWNsSet = false;
        }
    }
}

static void USBCtrlTrfInHandler(void)
{
    uint8_t lastDTS;

    lastDTS = pBDTEntryIn[0]->STAT.DTS;

    ((uint8_t_VAL*)&pBDTEntryIn[0])->Val ^= USB_NEXT_EP0_IN_PING_PONG;

    if(USBDeviceState == ADR_PENDING_STATE)
    {
        U1ADDR = (SetupPkt.bDevADR & 0x7F);
        if(U1ADDR != 0u)
        {
            USBDeviceState=ADDRESS_STATE;
        }
        else
        {
            USBDeviceState=DEFAULT_STATE;
        }
    }


    if(controlTransferState == CTRL_TRF_TX)
    {
        pBDTEntryIn[0]->ADR = ConvertToPhysicalAddress(CtrlTrfData);
        USBCtrlTrfTxService();

        if(shortPacketStatus == SHORT_PKT_SENT)
        {
            pBDTEntryIn[0]->STAT.Val = _BSTALL;
            pBDTEntryIn[0]->STAT.Val |= _USIE;
        }
        else
        {
            if(lastDTS == 0)
            {
                pBDTEntryIn[0]->STAT.Val = _DAT1|(_DTSEN & _DTS_CHECKING_ENABLED);
                pBDTEntryIn[0]->STAT.Val |= _USIE;
            }
            else
            {
                pBDTEntryIn[0]->STAT.Val = _DAT0|(_DTSEN & _DTS_CHECKING_ENABLED);
                pBDTEntryIn[0]->STAT.Val |= _USIE;
            }
        }
    }
	else 
    {
        if(outPipes[0].info.bits.busy == 1)
        {
            if(outPipes[0].pFunc != NULL)
            {
                outPipes[0].pFunc();
            }
            outPipes[0].info.bits.busy = 0;
        }
    	
        controlTransferState = WAIT_SETUP;
	}	

}

static void USBCheckStdRequest(void)
{
    if(SetupPkt.RequestType != USB_SETUP_TYPE_STANDARD_BITFIELD) return;

    switch(SetupPkt.bRequest)
    {
        case USB_REQUEST_SET_ADDRESS:
            inPipes[0].info.bits.busy = 1;            // This will generate a zero length packet
            USBDeviceState = ADR_PENDING_STATE;       
            break;
            
        case USB_REQUEST_GET_DESCRIPTOR:
            USBStdGetDscHandler();
            break;
            
        case USB_REQUEST_SET_CONFIGURATION:
            USBStdSetCfgHandler();
            break;
            
        case USB_REQUEST_GET_CONFIGURATION:
            inPipes[0].pSrc.bRam = (uint8_t*)&USBActiveConfiguration;         
            inPipes[0].info.bits.ctrl_trf_mem = USB_EP0_RAM;               
            inPipes[0].wCount.v[0] = 1;                         
            inPipes[0].info.bits.busy = 1;
            break;
            
        case USB_REQUEST_GET_STATUS:
            USBStdGetStatusHandler();
            break;
            
        case USB_REQUEST_CLEAR_FEATURE:
        case USB_REQUEST_SET_FEATURE:
            USBStdFeatureReqHandler();
            break;
            
        case USB_REQUEST_GET_INTERFACE:
            inPipes[0].pSrc.bRam = (uint8_t*)&USBAlternateInterface[SetupPkt.bIntfID];  
            inPipes[0].info.bits.ctrl_trf_mem = USB_EP0_RAM;              
            inPipes[0].wCount.v[0] = 1;                        
            inPipes[0].info.bits.busy = 1;
            break;
            
        case USB_REQUEST_SET_INTERFACE:
            inPipes[0].info.bits.busy = 1;
            USBAlternateInterface[SetupPkt.bIntfID] = SetupPkt.bAltID;
            break;
            
        case USB_REQUEST_SET_DESCRIPTOR:
            USB_SET_DESCRIPTOR_HANDLER(EVENT_SET_DESCRIPTOR,0,0);
            break;
            
        case USB_REQUEST_SYNCH_FRAME:
        default:
            break;
    }
}

static void USBStdFeatureReqHandler(void)
{
    BDT_ENTRY *p;
    EP_STATUS current_ep_data;
    unsigned char* pUEP;             

    if((SetupPkt.bFeature == USB_FEATURE_DEVICE_REMOTE_WAKEUP)&&
       (SetupPkt.Recipient == USB_SETUP_RECIPIENT_DEVICE_BITFIELD))
    {
        inPipes[0].info.bits.busy = 1;
        if(SetupPkt.bRequest == USB_REQUEST_SET_FEATURE)
            RemoteWakeup = true;
        else
            RemoteWakeup = false;
    }

    if((SetupPkt.bFeature == USB_FEATURE_ENDPOINT_HALT)&&
       (SetupPkt.Recipient == USB_SETUP_RECIPIENT_ENDPOINT_BITFIELD)&&
       (SetupPkt.EPNum != 0) && (SetupPkt.EPNum <= USB_MAX_EP_NUMBER)&&
       (USBDeviceState == CONFIGURED_STATE))
    {
		inPipes[0].info.bits.busy = 1;

        if(SetupPkt.EPDir == OUT_FROM_HOST)
        {
            p = (BDT_ENTRY*)pBDTEntryOut[SetupPkt.EPNum];
            current_ep_data.Val = ep_data_out[SetupPkt.EPNum].Val;
        }
        else
        {
            p = (BDT_ENTRY*)pBDTEntryIn[SetupPkt.EPNum];
            current_ep_data.Val = ep_data_in[SetupPkt.EPNum].Val;
        }

        #if (USB_PING_PONG_MODE == USB_PING_PONG__ALL_BUT_EP0) || (USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG)   
            if(current_ep_data.bits.ping_pong_state == 0)
            {
                USBHALPingPongSetToEven(&p);
            }
            else 
            {
                USBHALPingPongSetToOdd(&p);
            }
        #endif

        if(SetupPkt.EPDir == OUT_FROM_HOST)
        {
            pBDTEntryOut[SetupPkt.EPNum] = (volatile BDT_ENTRY *)p;
        }
        else
        {
            pBDTEntryIn[SetupPkt.EPNum] = (volatile BDT_ENTRY *)p;
        }

        if(SetupPkt.bRequest == USB_REQUEST_SET_FEATURE)
        {
            if(p->STAT.UOWN == 1)
            {
                if(SetupPkt.EPDir == OUT_FROM_HOST)
                {
                    ep_data_out[SetupPkt.EPNum].bits.transfer_terminated = 1;
                }
                else
                {
                    ep_data_in[SetupPkt.EPNum].bits.transfer_terminated = 1;
                }
            }
            p->STAT.Val |= _BSTALL;
            p->STAT.Val |= _USIE;
        }
        else
        {
            #if (USB_PING_PONG_MODE == USB_PING_PONG__ALL_BUT_EP0) || (USB_PING_PONG_MODE == USB_PING_PONG__FULL_PING_PONG)   
                USBAdvancePingPongBuffer(&p);  

                if(p->STAT.UOWN == 1)
                {
                    p->STAT.Val &= (~_USIE);   
                    p->STAT.Val |= _DAT1;      
                    USB_TRANSFER_TERMINATED_HANDLER(EVENT_TRANSFER_TERMINATED,p,sizeof(p));
                }
                else
                {
					p->STAT.Val |= _DAT1;
                }

                USBAdvancePingPongBuffer(&p);    

                if((current_ep_data.bits.transfer_terminated != 0) || (p->STAT.UOWN == 1))
                {
                    if(SetupPkt.EPDir == OUT_FROM_HOST)
                    {
                        ep_data_out[SetupPkt.EPNum].bits.transfer_terminated = 0;
                    }
                    else
                    {
                        ep_data_in[SetupPkt.EPNum].bits.transfer_terminated = 0;
                    }
                    p->STAT.Val &= ~(_USIE | _DAT1 | _BSTALL);  

                    USB_TRANSFER_TERMINATED_HANDLER(EVENT_TRANSFER_TERMINATED,p,sizeof(p));
                }
                else
                {
                    p->STAT.Val &= ~(_USIE | _DAT1 | _BSTALL); 
                } 
            #else /
                if((current_ep_data.bits.transfer_terminated != 0) || (p->STAT.UOWN == 1))
                {
                    if(SetupPkt.EPDir == OUT_FROM_HOST)
                    {
                        ep_data_out[SetupPkt.EPNum].bits.transfer_terminated = 0;
                    }
                    else
                    {
                        ep_data_in[SetupPkt.EPNum].bits.transfer_terminated = 0;
                    }
                    p->STAT.Val &= ~(_USIE | _BSTALL);  
                    p->STAT.Val |= _DAT1;                           
                    USB_TRANSFER_TERMINATED_HANDLER(EVENT_TRANSFER_TERMINATED,p,sizeof(p));
                }
                else
                {
                    p->STAT.Val &= ~(_USIE | _BSTALL);  
                    p->STAT.Val |= _DAT1;
                } 
            #endif 
            pUEP = (unsigned char*)(&U1EP0+SetupPkt.EPNum);
            *pUEP &= ~UEP_STALL;            
        }
    }
}
